# Vector procedures
# All procedures work on n-vectors except
# vecCross, matInvert, vecZero, matIdentity, matMake4, matRandomRot
# which assume 3-vectors.
# Author: Jeff Comer <jeffcomer at gmail>

proc vecZero {} {
    return [list 0.0 0.0 0.0]
}
proc vecInvert {a} {
    set b {}
    foreach ai $a {
	lappend b [expr {-$ai}]
    }
    return $b
}
proc vecAdd {a b} {
    set c {}
    foreach ai $a bi $b {
	lappend c [expr {$ai+$bi}]
    }
    return $c
}
proc vecSub {a b} {
    set c {}
    foreach ai $a bi $b {
	lappend c [expr {$ai-$bi}]
    }
    return $c
}
proc vecDot {a b} {
    set sum 0
    foreach ai $a bi $b {
	set sum [expr {$sum + $ai*$bi}]
    }
    return $sum
}
proc vecCross {a b} {
    foreach {ax ay az} $a {break}
    foreach {bx by bz} $b {break}
    
    set cx [expr {$ay*$bz - $az*$by}]
    set cy [expr {$az*$bx - $ax*$bz}]
    set cz [expr {$ax*$by - $ay*$bx}]
    return [list $cx $cy $cz]
}
proc vecLength {a} {
    set sum 0
    foreach ai $a {
	set sum [expr {$sum + $ai*$ai}]
    }
    return [expr sqrt($sum)]
}
proc vecLength2 {a} {
    set sum 0
    foreach ai $a {
	set sum [expr {$sum + $ai*$ai}]
    }
    return [expr $sum]
}
proc vecUnit {v} {
    set len [vecLength $v]
    return [vecScale [expr {1.0/$len}] $v]
}
proc vecScale {s a} {
    set b {}
    foreach ai $a {
	lappend b [expr {$ai*$s}]
    }
    return $b
}
proc vecTransform {m a} {
    set b {}
    foreach row $m {
	lappend b [vecDot $row $a]
    }
    return $b
}
proc vecRandom {} {
    set pi [expr {4.*atan(1.)}]
    # Create a random vector, uniform on a sphere.
    set theta [expr {2.0*rand()*$pi}]
    set phi [expr {acos(2.0*rand()-1.0)}]

    set x [expr {cos($theta)*sin($phi)}]
    set y [expr {sin($theta)*sin($phi)}]
    set z [expr {cos($phi)}]

    return [list $x $y $z]
}

proc dodecahedron {} {
    set phi [expr {0.5*(1.0 + sqrt(5.0))}]
    set invPhi [expr {1.0/$phi}]

    set vertexList {}
    foreach i {-1.0 1.0} {
	foreach j {-1.0 1.0} {
	    foreach k {-1.0 1.0} {
		lappend vertexList [list $i $j $k]
	    }
	}
    }

    foreach i {-1.0 1.0} {
	foreach j {-1.0 1.0} {
	    lappend vertexList [list 0.0 [expr {$i*$invPhi}] [expr {$j*$phi}]]
	    lappend vertexList [list [expr {$i*$invPhi}] [expr {$j*$phi}] 0.0]
	    lappend vertexList [list [expr {$j*$phi}] 0.0 [expr {$i*$invPhi}]]
	}
    }
    return $vertexList
}

proc icosahedron {} {
    set phi [expr {0.5*(1.0 + sqrt(5.0))}]
    
    set vertexList {}
    foreach i {-1.0 1.0} {
	foreach j {-1.0 1.0} {
	    lappend vertexList [list 0 $i [expr {$j*$phi}]]
	    lappend vertexList [list $i [expr {$j*$phi}] 0]
	    lappend vertexList [list [expr {$j*$phi}] 0 $i]
	}
    }
    return $vertexList
}

proc shiftWrap {v} {
    return [concat [lindex $v end] [lrange $v 0 end-1]]
}

proc truncIcosahedron {} {
    set phi [expr {0.5*(1.0 + sqrt(5.0))}]
    set threePhi [expr {3.0*$phi}]
    
    set vertexList {}
    foreach i {-1.0 1.0} {
	foreach j {-1.0 1.0} {
	    set v [list 0.0 $i [expr {$j*$threePhi}]]
	    lappend vertexList $v
	    lappend vertexList [shiftWrap $v]
	    lappend vertexList [shiftWrap [shiftWrap $v]]
	}
    }

    set permList {{0 1 2} {2 0 1} {}}
    foreach i {-1.0 1.0} {
	foreach j {-1.0 1.0} {
	    foreach k {-1.0 1.0} {
		set v1 [list [expr {$i*$phi}] [expr {$j*2.0}] [expr {$k*(1.0+2.0*$phi)}]]
		lappend vertexList $v1
		lappend vertexList [shiftWrap $v1]
		lappend vertexList [shiftWrap [shiftWrap $v1]]


		set v2 [list [expr {$i*2.0*$phi}] $j [expr {$k*(2.0+$phi)}]]
		lappend vertexList $v2
		lappend vertexList [shiftWrap $v2]
		lappend vertexList [shiftWrap [shiftWrap $v2]]
	    }
	}
    }
    return $vertexList
}


proc matIdentity {} {
    return [list [list 1.0 0.0 0.0] [list 0.0 1.0 0.0] [list 0.0 0.0 1.0]]
}
proc matTranspose {m} {
    set n [llength $m]
    set t $m

    for {set i 0} {$i < $n} {incr i} {
	for {set j 0} {$j < $n} {incr j} {
	    lset t $i $j [lindex $m $j $i]
	}
    }
    return $t
}
proc matScale {s m} {
    set ret {}
    
    foreach row $m {
	set newRow {}
	foreach n $row {
	    lappend newRow [expr {$s*$n}]
	}
	lappend ret $newRow
    }
    return $ret
}
proc matInvert {m} {
    foreach {mxx mxy mxz} [lindex $m 0] {break}
    foreach {myx myy myz} [lindex $m 1] {break}
    foreach {mzx mzy mzz} [lindex $m 2] {break}
    
    set det [expr {1.0*($mxx*($myy*$mzz-$myz*$mzy) - $mxy*($myx*$mzz-$myz*$mzx) + $mxz*($myx*$mzy-$myy*$mzx))}]
    set ixx [expr {($myy*$mzz - $myz*$mzy)/$det}]
    set ixy [expr {-($mxy*$mzz - $mxz*$mzy)/$det}]
    set ixz [expr {($mxy*$myz - $mxz*$myy)/$det}]
    set iyx [expr {-($myx*$mzz - $myz*$mzx)/$det}]
    set iyy [expr {($mxx*$mzz - $mxz*$mzx)/$det}]
    set iyz [expr {-($mxx*$myz - $mxz*$myx)/$det}]
    set izx [expr {($myx*$mzy - $myy*$mzx)/$det}]
    set izy [expr {-($mxx*$mzy - $mxy*$mzx)/$det}]
    set izz [expr {($mxx*$myy - $mxy*$myx)/$det}]

    return [list [list $ixx $ixy $ixz] [list $iyx $iyy $iyz] [list $izx $izy $izz]]
}
proc matDet {m} {
    foreach {mxx mxy mxz} [lindex $m 0] {break}
    foreach {myx myy myz} [lindex $m 1] {break}
    foreach {mzx mzy mzz} [lindex $m 2] {break}

    set det [expr {1.0*($mxx*($myy*$mzz-$myz*$mzy) - $mxy*($myx*$mzz-$myz*$mzx) + $mxz*($myx*$mzy-$myy*$mzx))}]
    return $det
}
proc matMul {a b} {
    set bt [matTranspose $b]

    set ret {}
    foreach rowA $a {
	set r {}
	foreach colB $bt {
	    lappend r [vecDot $rowA $colB]
	}
	lappend ret $r
    }
    return $ret
}
proc matRandomRot {} {
    set pi [expr {4.*atan(1.)}]
    # Create a random rotation matrix, uniform on a sphere.
    set a [expr {2.0*rand()*$pi}]
    set b [expr {acos(2.0*rand()-1.0)}]
    set c [expr {2.0*rand()*$pi}]

    set ca [expr {cos($a)}]
    set sa [expr {sin($a)}]
    set za [expr {-sin($a)}]
    set cb [expr {cos($b)}]
    set sb [expr {sin($b)}]
    set zb [expr {-sin($b)}]
    set cc [expr {cos($c)}]
    set sc [expr {sin($c)}]
    set zc [expr {-sin($c)}]

    set ta [list [list $ca $za 0] [list $sa $ca 0] [list 0 0 1]]
    set tb [list [list 1 0 0] [list 0 $cb $zb] [list 0 $sb $cb]]
    set tc [list [list $cc $zc 0] [list $sc $cc 0] [list 0 0 1]]

    set basis $ta
    set basis [matMul $tb $basis]
    set basis [matMul $tc $basis]
    return $basis
}
proc matMake4 {m {d {0.0 0.0 0.0}}} {
    set ret {}
    lappend ret [concat [lindex $m 0] [lindex $d 0]]
    lappend ret [concat [lindex $m 1] [lindex $d 1]]
    lappend ret [concat [lindex $m 2] [lindex $d 2]]
    lappend ret [list 0.0 0.0 0.0 1.0]
}
proc matConvert3To4 {m {d {0.0 0.0 0.0}}} {
    set ret {}
    lappend ret [concat [lindex $m 0] [lindex $d 0]]
    lappend ret [concat [lindex $m 1] [lindex $d 1]]
    lappend ret [concat [lindex $m 2] [lindex $d 2]]
    lappend ret [list 0.0 0.0 0.0 1.0]
}
proc matConvert4To3 {m} {
    set mat {}
    lappend mat [lrange [lindex $m 0] 0 2]
    lappend mat [lrange [lindex $m 1] 0 2]
    lappend mat [lrange [lindex $m 2] 0 2]
    set disp [list [lindex $m 0 3] [lindex $m 1 3] [lindex $m 2 3]]
    return [list $mat $disp]
}
proc matBasisVec {m dir} {
    set ret {}
    foreach row $m {
	lappend ret [lindex $row $dir]
    }
    return $ret
}

