#!/usr/bin/tclsh
# Author: Jeff Comer <jeffcomer at gmail>

if {[llength $argv] != 5} {
    puts "Usage: tclsh checkAbf.tcl anchor0 anchor1 colvarHistGrad colvarHistCount outName"
    exit
}

# Input:
foreach {anchor0 anchor1 colvarHistGrad colvarHistCount outName} $argv { break }

proc writeData {fileName data} {
    set out [open $fileName w]
    foreach d $data {
	puts $out $d
    }
    close $out
}

proc getNumFrames {colvarHistFile} {
    # We assume that the frames have a header of two comment lines.
    set count 0
    set in [open $colvarHistFile r]
    while {[gets $in lin] >= 0} {
	if {[string match "#*" $lin]} {
	    # Keep track of how many headers we've seen.
	    # Get the next comment line as well so it doesn't get counted.
	    set lin1 [gets $in lin]
	    # We are on the next frame.
	    incr count
	}
    }
    close $in
    
    return $count
}

proc getFrame {colvarHistFile frame} {
    set count 0
    set writing 0
    set in [open $colvarHistFile r]
    set data {}
    while {[gets $in lin] >= 0} {
	if {$writing} {
	    if {[string match "#*" $lin]} {
		# We have reached the end of this frame.
		set writing 0
		break
	    } else {
		# Add to the data.
		if {[string length [string trim $lin]] > 0} {
		    #puts $lin
		    lappend data [concat $lin]
		}
	    }
	} elseif {[string match "#*" $lin]} {
	    # Keep track of how many headers we've seen.

	    # Are we at the requested frame.
	    if {$frame >= 0 && $frame == $count} {
		puts "Found frame $frame." 
		# This is the frame that we want.
		set writing 1
	    }
	    # Get the next comment line as well so it doesn't get counted.
	    set lin1 [gets $in lin]
	    # We are on the next frame.
	    incr count
	}
    }
    close $in

    return $data
}

proc meanData {data x0 x1} {
    set n 0
    set sum 0.0
    foreach d $data {
	foreach {x y} $d { break }
	if {$x < $x0 || $x >= $x1} { continue }

	set sum [expr {$sum + $y}]
	incr n
    }

    return [expr {$sum/$n}]
}

proc shiftData {data shift} {
    set ret {}
    foreach d $data {
	foreach {x y} $d { break }
	lappend ret [list $x [expr {$y+$shift}]]
    }
    return $ret
}

proc integrateGrad {gradList anchor0 anchor1} {
    set binDel [expr {[lindex $gradList 1 0]-[lindex $gradList 0 0]}]

    set pmfList {}
    set sum 0.0
    foreach item $gradList {
	foreach {z gradMean} $item { break }
	set sum [expr {$sum + $gradMean*$binDel}]
	set z0 [expr {$z + 0.5*$binDel}]

	lappend pmfList [list $z $sum]
    }

    set anchorLevel [meanData $pmfList $anchor0 $anchor1]
    return [shiftData $pmfList [expr {-$anchorLevel}]]
}

set frames [getNumFrames $colvarHistGrad]
set endFrame [expr {$frames-1}]
puts "frames $frames"

set midFrame [expr {$frames/2}]
set endGradList [getFrame $colvarHistGrad $endFrame]
set midGradList [getFrame $colvarHistGrad $midFrame]
set begGradList [getFrame $colvarHistGrad 0]
set endCountList [getFrame $colvarHistCount $endFrame]
set midCountList [getFrame $colvarHistCount $midFrame]
set begCountList [getFrame $colvarHistCount 0]

set firstCountList {}
set firstGradList {}
foreach midGrad $midGradList midCount $midCountList begGrad $begGradList begCount $begCountList {
    set x [lindex $midCount 0]
    set countF [lindex $midCount 1]
    set count0 [lindex $begCount 1]
    set gradF [lindex $midGrad 1]
    set grad0 [lindex $begGrad 1]

    set count1 [expr {$countF - $count0}]
    lappend firstCountList [list $x $count1]

    if {$count1 == 0} {
	set grad1 0.0
    } else {
	set grad1 [expr {($countF*$gradF - $count0*$grad0)/$count1}]
    }
    lappend firstGradList [list $x $grad1]
}

set secondCountList {}
set secondGradList {}
foreach endGrad $endGradList endCount $endCountList midGrad $midGradList midCount $midCountList  {
    set x [lindex $endCount 0]
    set countF [lindex $endCount 1]
    set count0 [lindex $midCount 1]
    set gradF [lindex $endGrad 1]
    set grad0 [lindex $midGrad 1]

    set count1 [expr {$countF - $count0}]
    lappend secondCountList [list $x $count1]

    if {$count1 == 0} {
	set grad1 0.0
    } else {
	set grad1 [expr {($countF*$gradF - $count0*$grad0)/$count1}]
    }
    lappend secondGradList [list $x $grad1]
}

writeData $outName.0.count $firstCountList
writeData $outName.0.grad $firstGradList
writeData $outName.1.count $secondCountList
writeData $outName.1.grad $secondGradList

set firstPmf [integrateGrad $firstGradList $anchor0 $anchor1]
writeData $outName.0.pmf $firstPmf
set secondPmf [integrateGrad $secondGradList $anchor0 $anchor1]
writeData $outName.1.pmf $secondPmf
