# A set of parsing modules for PLOP interface
# 
# Created July06
# Lucy Forrest 
#
# usage: 
# $string,$string = setup_hom_mod($string,$string,$string,$string,$string,$string,$string,$string)
# (null) = setup_addh($string,$string,$string,$string)
# (null) = setup_emin_clash($string,$string,$string,$string,$string,$string)
# (null) = setup_sc($string,$string,$string,$string,$string,$string,$string)
# (null) = setup_sc_scap($string,$string,$string,$string,$string,$string,$string)

use strict;
use lib "/razor/0/common/scripts/modules/parsing";

package   plop;
require   Exporter;
require   fasta;
require   formats;
my @ISA = qw (Exporter);
my @EXPORT = qw (setup_hom_mod setup_addh setup_emin_clash setup_sc setup_sc_scap);

#### EXPORTED ####

###############
# add hydrogens
###############
sub setup_addh {
	
	# all filenames, inpdb, datadir must exist, others are created  
	my ($inpdb,$outpdb,$confile,$logfile,$outdir,$datadir,$ions) = @_;
	my $stage = "addh";
    if(!$ions) { $ions = "no"; } # updated 02aug06
    
	&start_stage($stage,$inpdb,$outpdb,$outdir);

	### create control file - updated ions 02aug06
	my $text = &get_header($datadir,$logfile,$outdir);
	$text .= "
load pdb $inpdb sym none ions $ions water no opt no hetatm yes\n
minim hydrogens\n
eval ster\nener calc\n
write pdb $outpdb\n";
	###

	&mk_control_file($text, $confile);
	if (!-e $confile) { return 0; }
	return 1;

}

##################
# minimize clashes 
##################
sub setup_emin_clash {
	
	# provide a log file from plop adding hydrogens, or energy calculation that gives 
	# list of clashes = addh_output
	# all filenames, inpdb, addh_output,$datadir and get_clashes_exe must exist, others are created  
	my ($inpdb,$outpdb,$confile,$logfile,$addh_output,$get_clashes_exe,$outdir,$datadir,$ions) = @_;
	my $stage = "emin_clash";
	my $list  = "list.dat"; # output from $get_clashes_exe
	my $cutoff = 0.6;
    if (!$ions) { $ions = "no"; }

	&start_stage($stage,$inpdb,$outpdb,$outdir);
	if (!-e $addh_output) { print STDERR "Can't find $addh_output to create clash list\n"; return 0; }
	`$get_clashes_exe $addh_output $cutoff`;
	if (!-e $list) { 
	    print STDERR "Proplem with creating $list for minimizing clashes\n  Check $get_clashes_exe\n"; return 0; 
	}
	if (-z  $list) { 
		print STDERR "The file called $list is empty. No residues to minimize!\n"; 
	}
	
	### create control file - update ions 02aug06
	my $text = &get_header($datadir,$logfile,$outdir);
	$text .= "
load pdb $inpdb sym none ions $ions water no opt no  hetatm yes 
minim residues file $list 
eval ster\nener calc\n
write pdb $outpdb\n";
    ###

	&mk_control_file($text, $confile);
	if (!-e $confile) { return 0; }
	return 1;
}

######################
# sidechain prediction
# using scap list generated by nestscape.pl or run_nest+scap.pl
######################
sub setup_sc_scap {
	
	# all filenames, inpdb, scap_sc_list, datadir and get_clashes_exe must exist, others are created  
	my ($inpdb,$outpdb,$confile,$logfile,$scap_sc_list,$sc_list,$get_clashes_exe,$outdir,$datadir,$ions) = @_;
	my $stage = "sc";
	my (@uncons_res, @res_in_pdb);
    if (!$ions) { $ions = "no"; } # update ions - 06aug06
	
	&start_stage($stage,$inpdb,$outpdb,$outdir);

	# prepare list of sidechains
	if (!-e $scap_sc_list) { die "Can't find list of unconserved sidechains:$scap_sc_list\n"; }
	@uncons_res = `cat $scap_sc_list | sed 's/,[A-Z]//' | sed 's/,//'`;

	# create new list 
	open (LIST, ">$sc_list");

	# find unconserved residues (that haven't been chopped out of model)
	# update this to read all chains in pdb - 02aug06

#old
#	@res_in_pdb = `grep ' CA ' $inpdb`;
#	foreach my $res (@res_in_pdb) {
#	   my $resnum =  substr($res,22,4);

	open (PDB, $inpdb);
	while (<PDB>) { 
		my $line = $_;
		chomp($line);
		next unless (($line =~ /^ATOM/) or ($line =~ /^HETATM/));
		if ($line =~ / CA /) { 
			my $line = $_;
			my $resnum = substr($line,22,4);
			my $chain  = substr($line,21,1);
			if ($chain eq " ") { $chain = "_"; }
			$resnum =~ s/ //g; # remove spaces
			foreach my $r (@uncons_res) { 
				chomp($r);
     			if ($r == $resnum) { print LIST "$chain:$r\n"; last; }
   			} # 
		} # next atom
	}
	close LIST;
	if (-z $sc_list) { die "New list of unconserved sidechains is empty: $sc_list\n"; }
	
	### create control file -update ions 02aug06
	my $text = &get_header($datadir,$logfile,$outdir);
	$text .= "
load pdb $inpdb sym none ions $ions water no opt no hetatm yes  
side predict file $sc_list ofac_init 0.75 ofac_min 0.50 iter 5 rand yes failsafe yes\n
eval ster
ener calc\n
write pdb $outpdb\n";
	###

	&mk_control_file($text, $confile);
	if (!-e $confile) { return 0; }
	return 1;
}

######################
# sidechain prediction
# using ready-made list (generated by uncons2plop.pl)
######################
sub setup_sc {
	
	# all filenames, inpdb, sc_list, datadir must exist, others are created  
	my ($inpdb,$outpdb,$confile,$logfile,$sc_list,$outdir,$datadir,$ions) = @_;
	my $stage = "sc";
	my (@uncons_res, @res_in_pdb);
    if (!$ions) { $ions = "no"; } # update ions - 06aug06
	
	&start_stage($stage,$inpdb,$outpdb,$outdir);

	# prepare list of sidechains
	if (!-e $sc_list) { die "Can't find list of unconserved sidechains:$sc_list\n"; }
	
	### create control file
	my $text = &get_header($datadir,$logfile,$outdir);
	$text .= "
load pdb $inpdb sym none ions $ions water no opt no hetatm yes  
side predict file $sc_list ofac_init 0.75 ofac_min 0.50 iter 5 rand yes failsafe yes\n
eval ster
ener calc\n
write pdb $outpdb\n";
	###

	&mk_control_file($text, $confile);
	if (!-e $confile) { return 0; }
	return 1;
	
}
#########################
# for a sequence and one template
# and an input alignment
# creates PLOP alignment file
#########################
sub setup_hom_mod {

    # model name, query, template, fasta file name, directory name, parameter directory, log file name, chain id
	my ($model,$n1,$n2,$in_fasta,$out_dir,$plop_params,$log_file,$chain) = @_;
	my %pdbfile; my $text = ""; 
	my $alnfile = "$out_dir/$model.plop";
	
	# store alignment and remove slashes - plop can't read more than one chain anyway!
	my $fa = "";
	open (FA, $in_fasta); 
	while(<FA>) { s#\/##; $fa .= $_; } 
	close (FA);
	my %stored = &fasta::toFastaHash($fa);

	$pdbfile{$n2} = "$out_dir/$n2.pdb"; #template

    # create control file - fix CHAIN!
	$text = "
# plop control file for model $model
file datadir $plop_params
file outdir $out_dir
file logfile $log_file\n
homolog single &
    $alnfile &
    $pdbfile{$n2} &
    ssgaps yes &
    chain $chain &
    conserve yes &
    x2p yes &
    g2x yes &
    cterm no &
    nterm no &
    sym none &
    tail aligned\n
write pdb $model.pdb\n";
  #from cinque
  #echo "load pdb $PDB/16pk.pdb " >> $Homology.con
  #echo "homolog load 16pk.align 16pk.pdb &" >> $Homology.con
  #echo "native 16pk.pdb &" >> $Homology.con
  #echo "finalfile 16pk_homo.rmsd " >> $Homology.con
  #echo "structure write 16pk_homo.pdb" >> $Homology.con
  #initfile file &
  #gapfile file &
  #insertfile file &
  #x2pfile file &
  #g2xfile file &
  #sidefile file &
  #finalfile file
  #het 

	# create Plop alignment file
	open (ALI, ">$alnfile");			
	print ALI &formats::toPlopFormat ($n1,\%stored);
    
	# output = filename for alignment, string contents of control file
	return ($alnfile, $text);
}




############################################################################
# end modules
############################################################################

###################
# local subroutines
###################
sub start_stage {

   my ($stage,$inpdb,$outpdb) = @_;
   if (-e "$outpdb") { print STDERR "Found $outpdb. Overwriting.\n"; }
   if (!-e "$inpdb") { print STDERR "ERROR! Can't find input pdb $inpdb! in start_stage plop.pm\n"; exit; }

   my $time = `date`; chomp($time);
   print STDERR "Creating input for $stage on $inpdb at $time...\n";
   
}

##################
sub get_header { 

   my ($datadir,$logfile,$outdir) = @_;
   my $header = "
file datadir $datadir
file outdir $outdir
file logfile $logfile\n\n";
   return ($header);

}

##################
sub mk_control_file {

   my ($txt, $confile) = @_;
 
   open (CONFILE, ">$confile") or die "Couldn't create $confile with $txt\n";
   print CONFILE "$txt";
   close (CONFILE);

}

